#ifndef __CWindowTools__
#define __CWindowTools__

#include <Basics/CString.hpp>
#include <MathTools/CRect.hpp>
#include <MathTools/CPoint.hpp>
#include <MathTools/CDimension.hpp>
#include "SWindowHandle.hpp"

//	===========================================================================

using Exponent::Basics::CString;
using Exponent::MathTools::CRect;
using Exponent::MathTools::CPoint;
using Exponent::MathTools::CDimension;
using Exponent::GUI::Windowing::SWindowHandle;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Windowing
		{
			/**
			 * @class CWindowTools CWindowTools.hpp
			 * @brief Lots of tools for dealing with windows
			 *
			 * @date 24/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CWindowTools.hpp,v 1.7 2007/02/27 19:47:11 paul Exp $
			 */
			class CWindowTools
			{
			public:

				/**
				 * Get the desktop window
				 * @param handle On return points to the handle of the desktop window
				 */
				static void getDesktopWindow(SWindowHandle *handle);

				/**
				 * Get the foreground window
				 * @param handle On return points to the handle of the forground window
				 */
				static void getForeGroundWindow(SWindowHandle *handle);

				/**
				 * Set the foreground window
				 * @param windowHandle The window handle to make foreground
				 */
				static void setForeGroundWindow(SWindowHandle *windowHandle);

				/**
				 * Set the focus'd window
				 * @param windowHandle The window handle to make focussed window
				 */
				static void setFocusWindow(SWindowHandle *windowHandle);

				/**
				 * Get the window under a point
				 * @param handle On return contains the window handle
				 * @param point The position to find the window under
				 */
				static void getWindowFromPoint(SWindowHandle *handle, const CPoint &point);

				/**
				 * Is the window visible
				 * @param windowHandle The window handle
				 * @retval bool True if the window is visible, false otherwise
				 */
				static bool isWindowVisible(SWindowHandle *windowHandle);

				/**
				 * Whats the mouse position
				 * @param point On return is filled with the specified position
				 */
				static void getMousePosition(CPoint &point);

				/**
				 * Set the cursor position
				 * @param point The mouse position
				 */
				static void setMousePosition(const CPoint &point);

				/**
				 * Compute the area that a string takes up
				 * @param drawContext On windows an HDC, on mac should be a font
				 * @param string The string to compute the size of
				 * @param dimension On return contains the string dimension
				 * @param winFont The windows font (HFONT), must be valid on windows, must be NULL on mac!
				 * @retval bool True if computed properly, false on error
				 */
				static bool computeStringArea(
									#ifndef WIN32
											  const 
									#endif 
											  void *drawContext, const CString &string, CDimension &dimension, void *winFont = NULL);

#ifdef WIN32
				/**
				 * Compute the string area, with no GDI+ startup routines.
				 * @note This function expects you to have setup the GDI+ startup stuff, You should use this function if you are computing lots of strings\n
				 * sizes sequentially (for example within a loop). Only to be used if you are sure you have GDI+
				 * @see GdiplusStartup
				 * @param drawContext The window HDC
				 * @param string The string to size
				 * @param dimension On output the size of the string
				 * @param font The HFONT for the font used to display the string
				 */
				static bool computeStringAreaNoGDIPlusStartup(HDC drawContext, const CString &string, CDimension &dimension, HFONT font);
#endif

				/**
				 * Get the window area
				 * @param windowHandle Handle for the window
				 * @param areaToFill On return contains the window area
				 * @retval bool True if filled properly, false otherwise
				 */
				static bool getWindowArea(SWindowHandle *windowHandle, CRect &areaToFill);

				/**
				 * Get the dimensions of the content area (the area you can draw in)
				 * @param windowHandle Handle for the window
				 * @param dimension On return is filled with window area
				 * @retval bool True if filled properly, false otherwise
				 */
				static bool getContentArea(SWindowHandle *windowHandle, CDimension &dimension);

				/**
				 * Compute the area actually required including extended styled border (if extended style is not required, simply set to -1, will convert\n
				 * dimension to dimensionToFill, you should use dimensionToFill to create the window on mac simply copies the dimensions, as the sizing is done for you
				 * @param dimension The dimension you want
				 * @param dimensionToFill On return holds the new dimension
				 * @param style The window style
				 * @param exStyle The extended style
				 * @param hasMenu Does the window have a menu
				 * @retval bool True if filled properly, false otherwise
				 */
				static bool computeCorrectedWindowArea(const CDimension &dimension, CDimension &dimensionToFill, unsigned long style, unsigned long exStyle = -1, const bool hasMenu = false);

				/**
				 * Get the client area In screen coordinates
				 * @param windowHandle The handle of the window
				 * @param areaToFill On return contains the screen co-ordinates of the window handle
				 * @retval bool True if filled properly, false otherwise
				 */
				static bool getGlobalContentAreaCoordinates(SWindowHandle *windowHandle, CRect &areaToFill);

				/**
				 * Compute the logical size from a point size. No equivalent for mac
				 * @param pointSize The point size
				 * @retval long The logical size
				 */
				static long computeLogicalSizeFromPointSize(const long pointSize);
				
#ifndef WIN32
				/**
				 * Is this window a composited window
				 * @param window The window
				 * @retval bool True if composited, false otherwise
				 */
				static bool isWindowComposited(SWindowHandle *windowHandle);
#endif
			};
		}
	}
}
#endif	// End of CWindowTools.hpp